#ifndef __IImage__
#define __IImage__

#include <Basics/ICountedObject.hpp>
#include <Basics/CSystemString.hpp>
#include <MathTools/CDimension.hpp>
#include <MathTools/CRect.hpp>

//	===========================================================================

using Exponent::Basics::ICountedObject;
using Exponent::Basics::CSystemString;
using Exponent::MathTools::CDimension;
using Exponent::MathTools::CRect;

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Graphics
		{
			/**
			 * @interface IImage IImage.hpp
			 * @brief Interface to an image
			 *
			 * @date 19/03/2005
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 * @version 1.1.0 Made inherit from ICountedObject to preserve virtual IInage nature
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: IImage.hpp,v 1.6 2007/02/11 02:16:22 paul Exp $
			 */
			interface IImage : public ICountedObject
			{
			public:

//	===========================================================================

				/**
				 * @enum EImageFormats
				 * @brief The types of image format
				 */
				enum EImageFormats
				{
					e_bitmap = 0,	/**< Reserved for future use */
					e_targa,		/**< 32bpp Targa file format */
					e_png,			/**< Reserved for future use */
					e_jpeg,			/**< JPEG type file format */
					e_tiff,			/**< Reserved for future use */
					e_imageIcon,	/**< Small image, with constrained sizes */
				};

//	===========================================================================

				/**
				 * Virtual destructor
				 */
				virtual ~IImage() {}

//	===========================================================================

				/**
				 * Get the size of the image
				 * @retval const CDimension& The size of the image
				 */
				virtual const CDimension &getImageSize() const = 0;

				/**
				 * Get the normalised image size
				 * @retval const CRect& 0,0 original size
				 */
				virtual const CRect &getNormalisedImageSize() const = 0;

				/**
				 * Get the format
				 * @retval EImageFormats The format of the image
				 */
				virtual EImageFormats getImageFormat() const = 0;

				/**
				 * Get the file name of the image
				 * @retval const CSystemString& The name of the file that is currently loaded
				 */
				virtual const CSystemString &getFileName() const = 0;

//	===========================================================================

				/**
				 * Load from a file
				 * @param filename The name of the file
				 * @retval bool True if loaded the image properly, false on error
				 */
				virtual bool loadFromFile(const CSystemString &filename) = 0;

				/**
				 * Unload the image
				 */
				virtual void unloadImage() = 0;

//	===========================================================================

				/**
				 * Does the image have an alpha channel
				 * @retval bool True if the image has an alpha channel, false otherwise
				 */
				virtual bool imageHasAlpha() const = 0;

				/**
				 * Is there an image loaded
				 * @retval bool True if image is loaded, false otherwise
				 */
				virtual bool hasImageLoaded() const = 0;

//	===========================================================================

				/** @cond */
				#ifdef WIN32
				/** @endcond */
					/**
					 * Get the image handle
					 * @retval HBITMAP The windows bitmap
					 */
					virtual HBITMAP getImageHandle() const = 0;
				/** @cond */
				#else
				/** @endcond */
					/**
					 * Get the image handle
					 * @retval CGImageRef The mac image ref
					 */
					virtual CGImageRef getImageHandle() const = 0;
				/** @cond */
				#endif
				/** @endcond */
			};
		}
	}
}
#endif	// End of IImage.hpp